#!/bin/bash

# Copyright 2018 The Chromium OS Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

. "$(dirname "$0")/common.sh"

#######################################
# Return name according to the current signing debug key. The name is used to
# select key files.
# Globals:
#   None
# Arguments:
#   sha1: signature of the APK.
#   keyset: "cheets" or "aosp" build?
# Outputs:
#   Writes the name of the key to stdout.
# Returns:
#   0 on success, non-zero on error.
#######################################
android_choose_key() {
  local sha1="$1"
  local keyset="$2"

  if [[ "${keyset}" != "aosp" && "${keyset}" != "cheets" ]]; then
    error "Unknown Android build keyset '${keyset}'."
    return 1
  fi

  # Fingerprints below are generated by:
  # 'cheets' keyset:
  # $ keytool -file vendor/google_arc/certs/cheetskeys/$NAME.x509.pem \
  #     -printcert | grep SHA1:
  # 'aosp' keyset:
  # $ keytool -file build/target/product/security/$NAME.x509.pem -printcert \
  #     | grep SHA1:
  declare -A platform_sha=(
    ['cheets']='AA:04:E0:5F:82:9C:7E:D1:B9:F8:FC:99:6C:5A:54:43:83:D9:F5:BC'
    ['aosp']='27:19:6E:38:6B:87:5E:76:AD:F7:00:E7:EA:84:E4:C6:EE:E3:3D:FA'
  )
  declare -A media_sha=(
    ['cheets']='D4:C4:2D:E0:B9:1B:15:72:FA:7D:A7:21:E0:A6:09:94:B4:4C:B5:AE'
    ['aosp']='B7:9D:F4:A8:2E:90:B5:7E:A7:65:25:AB:70:37:AB:23:8A:42:F5:D3'
  )
  declare -A shared_sha=(
    ['cheets']='38:B6:2C:E1:75:98:E3:E1:1C:CC:F6:6B:83:BB:97:0E:2D:40:6C:AE'
    ['aosp']='5B:36:8C:FF:2D:A2:68:69:96:BC:95:EA:C1:90:EA:A4:F5:63:0F:E5'
  )
  declare -A release_sha=(
    ['cheets']='EC:63:36:20:23:B7:CB:66:18:70:D3:39:3C:A9:AE:7E:EF:A9:32:42'
    ['aosp']='61:ED:37:7E:85:D3:86:A8:DF:EE:6B:86:4B:D8:5B:0B:FA:A5:AF:81'
  )
  declare -A networkstack_sha=(
    ['cheets']='7C:AD:D6:52:41:69:E7:A4:47:6F:DA:74:D0:8E:F0:48:3A:6F:00:ED'
    ['aosp']='7C:8B:DA:BD:21:F9:53:A1:B1:8C:CB:E7:B9:13:93:D9:FD:F9:48:30'
  )

  case "${sha1}" in
    "${platform_sha["${keyset}"]}")
      echo "platform"
      ;;
    "${media_sha["${keyset}"]}")
      echo "media"
      ;;
    "${shared_sha["${keyset}"]}")
      echo "shared"
      ;;
    "${release_sha["${keyset}"]}")
      # The release_sha[] fingerprint is from devkey. Translate to releasekey.
      echo "releasekey"
      ;;
    "${networkstack_sha["${keyset}"]}")
      echo "networkstack"
      ;;
    *)
      # Not a framework apk.  Do not re-sign.
      echo ""
      ;;
  esac
  return 0
}

#######################################
# Extract 'ro.build.flavor' property from build property file.
# Globals:
#   None
# Arguments:
#   build_prop_file: path to build property file.
# Outputs:
#   Writes the value of the property to stdout.
# Returns:
#   0 on success, non-zero on error.
#######################################
android_get_build_flavor_prop() {
  local build_prop_file="$1"
  local flavor_prop=""

  if ! flavor_prop=$(grep -a "^ro\.build\.flavor=" "${build_prop_file}"); then
    return 1
  fi
  flavor_prop=$(echo "${flavor_prop}" | cut -d "=" -f2)
  echo "${flavor_prop}"
  return 0
}

#######################################
# Pick the expected keyset ('cheets', 'aosp') depending on the build flavor.
# Globals:
#   None
# Arguments:
#   flavor_prop: the value of the build flavor property.
# Outputs:
#   Writes the name of the keyset to stdout.
# Returns:
#   0 on success, non-zero on error.
#######################################
android_choose_signing_keyset() {
  local flavor_prop="$1"

  # Property ro.build.flavor follows those patterns:
  # - cheets builds:
  #   ro.build.flavor=cheets_${arch}-user(debug)
  # - SDK builds:
  #   ro.build.flavor=sdk_google_cheets_${arch}-user(debug) # For N
  #   ro.build.flavor=sdk_cheets_${arch}-user(debug) # For P
  # - AOSP builds:
  #   ro.build.flavor=aosp_cheets_${arch}-user(debug)
  # "cheets" and "SDK" builds both use the same signing keys, cheetskeys. "AOSP"
  # builds use the public AOSP signing keys.
  if [[ "${flavor_prop}" == aosp_cheets_* ]]; then
    keyset="aosp"
  elif [[ "${flavor_prop}" == cheets_* ||
    "${flavor_prop}" == sdk_cheets_* ||
    "${flavor_prop}" == sdk_google_cheets_* ||
    "${flavor_prop}" == bertha_* ]]; then
    keyset="cheets"
  else
    return 1
  fi
  echo "${keyset}"
  return 0
}
